/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.imap.encode;

import com.github.fge.lambdas.Throwing;
import jakarta.inject.Inject;
import jakarta.mail.Flags;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.TreeSet;
import java.util.function.Consumer;
import org.apache.james.imap.api.ImapConstants;
import org.apache.james.imap.encode.EncoderUtils;
import org.apache.james.imap.encode.ImapResponseComposer;
import org.apache.james.imap.encode.ImapResponseEncoder;
import org.apache.james.imap.message.response.FetchResponse;
import org.apache.james.mailbox.MessageSequenceNumber;
import org.apache.james.mailbox.MessageUid;
import org.apache.james.mailbox.ModSeq;
import org.apache.james.mailbox.model.MessageId;
import org.apache.james.mailbox.model.ThreadId;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FetchResponseEncoder
implements ImapResponseEncoder<FetchResponse> {
    private static final Logger LOGGER = LoggerFactory.getLogger(FetchResponseEncoder.class);
    private static final byte[] ENVELOPE = "ENVELOPE".getBytes(StandardCharsets.US_ASCII);
    private static final byte[] RFC_822_SIZE = "RFC822.SIZE".getBytes(StandardCharsets.US_ASCII);
    private static final byte[] INTERNALDATE = "INTERNALDATE".getBytes(StandardCharsets.US_ASCII);
    private final boolean neverAddBodyStructureExtensions;

    @Inject
    public FetchResponseEncoder() {
        this(false);
    }

    public FetchResponseEncoder(boolean neverAddBodyStructureExtensions) {
        this.neverAddBodyStructureExtensions = neverAddBodyStructureExtensions;
    }

    @Override
    public Class<FetchResponse> acceptableMessages() {
        return FetchResponse.class;
    }

    @Override
    public void encode(FetchResponse fetchResponse, ImapResponseComposer composer) throws IOException {
        MessageSequenceNumber messageNumber = fetchResponse.getMessageNumber();
        composer.untagged().message(messageNumber.asInt()).message(ImapConstants.FETCH_COMMAND.getNameAsBytes()).openParen();
        this.encodeModSeq(composer, fetchResponse);
        this.encodeFlags(composer, fetchResponse);
        this.encodeInternalDate(composer, fetchResponse);
        this.encodeSize(composer, fetchResponse);
        this.encodeEnvelope(composer, fetchResponse);
        this.encodeBody(composer, fetchResponse.getBody());
        this.encodeBodyStructure(composer, fetchResponse.getBodyStructure());
        this.encodeUid(composer, fetchResponse);
        this.encodeBodyElements(composer, fetchResponse.getElements());
        this.encodeEmailId(composer, fetchResponse);
        this.encodeThreadId(composer, fetchResponse);
        this.encodeSaveDate(composer, fetchResponse);
        composer.closeParen().end();
    }

    private void encodeModSeq(ImapResponseComposer composer, FetchResponse response) throws IOException {
        ModSeq modSeq = response.getModSeq();
        if (modSeq != null) {
            composer.message(ImapConstants.FETCH_MODSEQ);
            composer.openParen();
            composer.skipNextSpace();
            composer.message(modSeq.asLong());
            composer.closeParen();
        }
    }

    private void encodeBody(ImapResponseComposer composer, FetchResponse.Structure body) throws IOException {
        if (body != null) {
            composer.message(ImapConstants.FETCH_BODY);
            this.encodeStructure(composer, body, false, false);
        }
    }

    private void encodeBodyStructure(ImapResponseComposer composer, FetchResponse.Structure bodyStructure) throws IOException {
        if (bodyStructure != null) {
            composer.message(ImapConstants.FETCH_BODY_STRUCTURE);
            this.encodeStructure(composer, bodyStructure, true, false);
        }
    }

    private void encodeStructure(ImapResponseComposer composer, FetchResponse.Structure structure, boolean includeExtensions, boolean isInnerPart) throws IOException {
        String subType;
        String mediaType;
        String rawMediaType = structure.getMediaType();
        if (rawMediaType == null) {
            mediaType = "TEXT";
            subType = "PLAIN";
        } else {
            mediaType = rawMediaType;
            subType = structure.getSubType();
        }
        this.encodeStructure(composer, structure, includeExtensions, mediaType, subType, isInnerPart);
    }

    private void encodeStructure(ImapResponseComposer composer, FetchResponse.Structure structure, boolean includeExtensions, String mediaType, String subType, boolean isInnerPart) throws IOException {
        boolean includeOptionalExtensions;
        boolean bl = includeOptionalExtensions = includeExtensions && !this.neverAddBodyStructureExtensions;
        if (isInnerPart) {
            composer.skipNextSpace();
        }
        if ("MULTIPART".equalsIgnoreCase(mediaType)) {
            this.encodeMultipart(composer, structure, subType, includeOptionalExtensions);
        } else if ("MESSAGE".equalsIgnoreCase(mediaType) && "RFC822".equalsIgnoreCase(subType)) {
            this.encodeRfc822Message(composer, structure, mediaType, subType, includeOptionalExtensions);
        } else {
            this.encodeBasic(composer, structure, includeOptionalExtensions, mediaType, subType);
        }
    }

    private void encodeBasic(ImapResponseComposer composer, FetchResponse.Structure structure, boolean includeExtensions, String mediaType, String subType) throws IOException {
        if ("TEXT".equalsIgnoreCase(mediaType)) {
            long lines = structure.getLines();
            this.encodeBodyFields(composer, structure, mediaType, subType);
            composer.message(lines);
        } else {
            this.encodeBodyFields(composer, structure, mediaType, subType);
        }
        if (includeExtensions) {
            this.encodeOnePartBodyExtensions(composer, structure);
        }
        composer.closeParen();
    }

    private void encodeOnePartBodyExtensions(ImapResponseComposer composer, FetchResponse.Structure structure) throws IOException {
        String md5 = structure.getMD5();
        List<String> languages = structure.getLanguages();
        String location = structure.getLocation();
        this.nillableQuote(composer, md5);
        this.bodyFldDsp(structure, composer);
        this.nillableQuotes(composer, languages);
        this.nillableQuote(composer, location);
    }

    private ImapResponseComposer bodyFldDsp(FetchResponse.Structure structure, ImapResponseComposer composer) throws IOException {
        String disposition = structure.getDisposition();
        if (disposition == null) {
            composer.nil();
        } else {
            composer.openParen();
            composer.quote(disposition);
            Map<String, String> params = structure.getDispositionParams();
            this.bodyFldParam(params, composer);
            composer.closeParen();
        }
        return composer;
    }

    private void bodyFldParam(Map<String, String> params, ImapResponseComposer composer) throws IOException {
        if (params == null || params.isEmpty()) {
            composer.nil();
        } else {
            composer.openParen();
            Set<String> keySet = params.keySet();
            TreeSet<String> names = new TreeSet<String>(keySet);
            for (String name : names) {
                String value = params.get(name);
                if (value == null) {
                    LOGGER.debug("Disposition parameter {} has no matching value", (Object)name);
                    continue;
                }
                composer.quote(name);
                composer.quote(value);
            }
            composer.closeParen();
        }
    }

    private void encodeBodyFields(ImapResponseComposer composer, FetchResponse.Structure structure, String mediaType, String subType) throws IOException {
        List<String> bodyParams = structure.getParameters();
        String id = structure.getId();
        String description = structure.getDescription();
        String encoding = structure.getEncoding();
        long octets = structure.getOctets();
        composer.openParen().quoteUpperCaseAscii(mediaType).quoteUpperCaseAscii(subType);
        this.nillableQuotes(composer, bodyParams);
        this.nillableQuote(composer, id);
        this.nillableQuote(composer, description);
        composer.quoteUpperCaseAscii(encoding).message(octets);
    }

    private void encodeMultipart(ImapResponseComposer composer, FetchResponse.Structure structure, String subType, boolean includeExtensions) throws IOException {
        composer.openParen();
        Iterator<FetchResponse.Structure> it = structure.parts();
        while (it.hasNext()) {
            FetchResponse.Structure part = it.next();
            this.encodeStructure(composer, part, includeExtensions, true);
        }
        composer.quoteUpperCaseAscii(subType);
        if (includeExtensions) {
            List<String> languages = structure.getLanguages();
            this.nillableQuotes(composer, structure.getParameters());
            this.bodyFldDsp(structure, composer);
            this.nillableQuotes(composer, languages);
            this.nillableQuote(composer, structure.getLocation());
        }
        composer.closeParen();
    }

    private void encodeRfc822Message(ImapResponseComposer composer, FetchResponse.Structure structure, String mediaType, String subType, boolean includeExtensions) throws IOException {
        long lines = structure.getLines();
        FetchResponse.Envelope envelope = structure.getEnvelope();
        FetchResponse.Structure embeddedStructure = structure.getBody();
        this.encodeBodyFields(composer, structure, mediaType, subType);
        this.encodeEnvelope(composer, envelope, false);
        this.encodeStructure(composer, embeddedStructure, includeExtensions, false);
        composer.message(lines);
        if (includeExtensions) {
            this.encodeOnePartBodyExtensions(composer, structure);
        }
        composer.closeParen();
    }

    private void encodeBodyElements(ImapResponseComposer composer, List<FetchResponse.BodyElement> elements) throws IOException {
        if (elements != null) {
            for (FetchResponse.BodyElement element : elements) {
                String name = element.getName();
                composer.message(name);
                composer.literal(element);
            }
        }
    }

    private void encodeSize(ImapResponseComposer composer, FetchResponse fetchResponse) throws IOException {
        Long size = fetchResponse.getSize();
        if (size != null) {
            composer.message(RFC_822_SIZE);
            composer.message(size.intValue());
        }
    }

    private void encodeInternalDate(ImapResponseComposer composer, FetchResponse fetchResponse) throws IOException {
        Date internalDate = fetchResponse.getInternalDate();
        if (internalDate != null) {
            composer.message(INTERNALDATE);
            composer.quote(EncoderUtils.encodeDateTime(internalDate));
        }
    }

    private void encodeSaveDate(ImapResponseComposer composer, FetchResponse fetchResponse) throws IOException {
        Optional<Date> saveDate = fetchResponse.getSaveDate();
        if (this.isSaveDateFetched(saveDate)) {
            composer.message("SAVEDATE");
            saveDate.ifPresentOrElse((Consumer<Date>)Throwing.consumer(date -> composer.quote(EncoderUtils.encodeDateTime(date))), (Runnable)Throwing.runnable(composer::nil));
        }
    }

    private boolean isSaveDateFetched(Optional<Date> saveDate) {
        return saveDate != null;
    }

    private void encodeUid(ImapResponseComposer composer, FetchResponse fetchResponse) throws IOException {
        MessageUid uid = fetchResponse.getUid();
        if (uid != null) {
            composer.message(ImapConstants.UID);
            composer.message(uid.asLong());
        }
    }

    private void encodeEmailId(ImapResponseComposer composer, FetchResponse fetchResponse) throws IOException {
        MessageId emailId = fetchResponse.getEmailId();
        if (emailId != null) {
            composer.message("EMAILID");
            composer.openParen();
            composer.message(emailId.serialize());
            composer.closeParen();
        }
    }

    private void encodeThreadId(ImapResponseComposer composer, FetchResponse fetchResponse) throws IOException {
        ThreadId threadId = fetchResponse.getThreadId();
        if (threadId != null) {
            composer.message("THREADID");
            composer.openParen();
            composer.message(threadId.serialize());
            composer.closeParen();
        }
    }

    private void encodeFlags(ImapResponseComposer composer, FetchResponse fetchResponse) throws IOException {
        Flags flags = fetchResponse.getFlags();
        if (flags != null) {
            composer.flags(flags);
        }
    }

    private void encodeEnvelope(ImapResponseComposer composer, FetchResponse fetchResponse) throws IOException {
        FetchResponse.Envelope envelope = fetchResponse.getEnvelope();
        this.encodeEnvelope(composer, envelope, true);
    }

    private void encodeEnvelope(ImapResponseComposer composer, FetchResponse.Envelope envelope, boolean prefixWithName) throws IOException {
        if (envelope != null) {
            String date = envelope.getDate();
            String subject = envelope.getSubject();
            FetchResponse.Envelope.Address[] from = envelope.getFrom();
            FetchResponse.Envelope.Address[] sender = envelope.getSender();
            FetchResponse.Envelope.Address[] replyTo = envelope.getReplyTo();
            FetchResponse.Envelope.Address[] to = envelope.getTo();
            FetchResponse.Envelope.Address[] cc = envelope.getCc();
            FetchResponse.Envelope.Address[] bcc = envelope.getBcc();
            String inReplyTo = envelope.getInReplyTo();
            String messageId = envelope.getMessageId();
            if (prefixWithName) {
                composer.message(ENVELOPE);
            }
            composer.openParen();
            this.nillableQuote(composer, date);
            this.nillableQuote(composer, subject);
            this.encodeAddresses(composer, from);
            this.encodeAddresses(composer, sender);
            this.encodeAddresses(composer, replyTo);
            this.encodeAddresses(composer, to);
            this.encodeAddresses(composer, cc);
            this.encodeAddresses(composer, bcc);
            this.nillableQuote(composer, inReplyTo);
            this.nillableQuote(composer, messageId);
            composer.closeParen();
        }
    }

    private void encodeAddresses(ImapResponseComposer composer, FetchResponse.Envelope.Address[] addresses) throws IOException {
        if (addresses == null || addresses.length == 0) {
            composer.nil();
        } else {
            composer.openParen();
            for (FetchResponse.Envelope.Address address : addresses) {
                this.encodeAddress(composer, address);
            }
            composer.closeParen();
        }
    }

    private void encodeAddress(ImapResponseComposer composer, FetchResponse.Envelope.Address address) throws IOException {
        String name = address.getPersonalName();
        String domainList = address.getAtDomainList();
        String mailbox = address.getMailboxName();
        String host = address.getHostName();
        composer.skipNextSpace().openParen();
        this.nillableQuote(composer, name);
        this.nillableQuote(composer, domainList);
        this.nillableQuote(composer, mailbox);
        this.nillableQuote(composer, host).closeParen();
    }

    private ImapResponseComposer nillableQuote(ImapResponseComposer composer, String message) throws IOException {
        if (message == null) {
            composer.nil();
        } else {
            composer.quote(message);
        }
        return composer;
    }

    private ImapResponseComposer nillableQuotes(ImapResponseComposer composer, List<String> quotes) throws IOException {
        if (quotes == null || quotes.size() == 0) {
            composer.nil();
        } else {
            composer.openParen();
            for (String string : quotes) {
                this.nillableQuote(composer, string);
            }
            composer.closeParen();
        }
        return composer;
    }
}

