/*
 * MapPage.java
 *
 * Brazil project web application toolkit,
 * export version: 2.1 
 * Copyright (c) 1999-2003 Sun Microsystems, Inc.
 *
 * Sun Public License Notice
 *
 * The contents of this file are subject to the Sun Public License Version 
 * 1.0 (the "License"). You may not use this file except in compliance with 
 * the License. A copy of the License is included as the file "license.terms",
 * and also available at http://www.sun.com/
 * 
 * The Original Code is from:
 *    Brazil project web application toolkit release 2.1.
 * The Initial Developer of the Original Code is: suhler.
 * Portions created by suhler are Copyright (C) Sun Microsystems, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s): cstevens, suhler.
 *
 * Version:  2.2
 * Created by suhler on 99/06/28
 * Last modified by suhler on 03/08/01 16:18:34
 */

package sunlabs.brazil.handler;

import java.util.Hashtable;
import java.util.Vector;
import java.util.Enumeration;
import sunlabs.brazil.util.regexp.Regexp;

/**
 * Utility class to rewrite links inside of web pages so they appear to
 * come from a different site.
 *
 * @author      Stephen Uhler
 * @version	@(#)MapPage.java	2.2
 */

public class MapPage {
    /**
     * Initialized to all tag/attribute pairs whose attribute values are
     * considered for rewriting.
     */
    public Hashtable tagMap;	// tag attributes to map
    public Hashtable urlMap;	// mapping table
    public Vector patternMap;	// mapping table for glob patterns
    public String prefix;	// prefix to prepend onto mapped url's
    				// that start with "/"
    public static boolean log = false;	// enable/disable diagnostics
    public int count;	// number of tags rewritten

    /**
     * Create a site mapper.  The following table maps all the entity/attribute
     * combinations that are (or could be) URL's to (possibly) rewrite.
     * @param prefix	Every link starting with "/" has the leading
     * 			slash replaced by <code>prefix</code>.  If prefix is
     *			<i>null</i>, then only fully qualified url's are
     *			considered for rewriting.
     */

    public MapPage(String prefix) {
    	this.prefix = prefix;
    	urlMap = null;
	patternMap = new Vector();
    	count = 0;
    	tagMap = new Hashtable(19);
        tagMap.put("a","href");
        tagMap.put("applet","codebase");
        tagMap.put("area","href");
        tagMap.put("base","href");
        tagMap.put("body","background");
        tagMap.put("embed","src"); 
        tagMap.put("form","action");
        tagMap.put("frame","src");
        tagMap.put("img","src");
        tagMap.put("layer","src");
        tagMap.put("link","href");
        tagMap.put("object","codebase");
        tagMap.put("param","value"); // Not sure about this one
        tagMap.put("td","background");
        tagMap.put("th","background");
        tagMap.put("input","src");	// image input items?
        tagMap.put("script","src");	// ???
        log("prefix: " + prefix);
    }

    /**
     * Change the prefix that will replace the leading "/" in a URL.
     */

    public void setPrefix(String prefix) {
        this.prefix = prefix;
        log("prefix: " + prefix);
    }

    /**
     * add a tag/attribute pair to the rewrite list.
     * If the attribute is null, remove the tag.
     */

    public void addTag(String name, String attribute) {
        if (attribute == null) {
	    tagMap.remove(name);
	} else {
	    tagMap.put(name.toLowerCase(), attribute.toLowerCase());
	}
    }

    /**
     * Set the map table for cross-linked sites.
     * Foreach key in the table that matches a URL, replace the key
     * portion of the url with the key's value.
     */

    public void setMap(Hashtable map) {
    	urlMap = map;
    }

    /**
     * Add or remove an item to the map table
     * @param name	The prefix of the url to match
     * @param value	The prefix to replace it with.  If null, 
     *			remove the prefix
     */

    public void
    addMapEntry(String name, String value) {
    	if (value == null && urlMap != null) {
	    urlMap.remove(name);
	    return;
    	} else if (urlMap == null) {
	    urlMap = new Hashtable();
	}
	urlMap.put(name, value);
    }

    /**
     * Add or remove an item to the pattern table
     * @param pattern	The prefix pattern of the url to match
     *			Full tcl8.0-style regexps are supported
     * @param replacement The prefix to replace it with.  If null, 
     *			remove the prefix.  \n's are replaced by the
     *			corrosponding sub-matches in the name
     * <P>
     * Patterns are stored in a vector, with each pattern taking 3
     * concecutive elements: the pattern, the replacement rule, and
     * the compiled expression.  This way they are searched in order.
     * Sample usage:
     *   http://www.([^.]*).com/     /site/\1/
     * will replace the url: http://www.foo.com/a/b.html with
     *   /site/foo/a/b.html
     */

    public void 
    addPatternEntry(String pattern, String replacement) {
	if (!pattern.startsWith("^")) {
	    pattern = "^" + pattern;
	}
	// System.out.println("Adding pattern: " + pattern + " (" + replacement + ")");
	int index = patternMap.indexOf(pattern);
	if (index > 0 && replacement == null) {
	    patternMap.removeElementAt(index);
	    patternMap.removeElementAt(index);
	    patternMap.removeElementAt(index);
	} else if (index > 0) {
	    patternMap.setElementAt(replacement, ++index);
	    patternMap.setElementAt(new Regexp(pattern), ++index);
	} else if (replacement != null) {
	    patternMap.addElement(pattern);
	    patternMap.addElement(replacement);
	    patternMap.addElement(new Regexp(pattern));
	}
    }

    /**
     * How many tags have been mapped?
     */

    public int mapCount() {
    	return count;
    }

    /**
     * Rewrite all the url's in this document.  This is accomplished
     * via repeated calls to {@link #convertString}.
     *
     * @param content	The HTML to be processed.
     * @return		The smae HTML, will all URL's rewritten.
     *			URL's starting with "/" have the "/" replaced
     *			with the prefix.  All others are re-written based
     *			on the supplied mapping tables.
     */

    public String convertHtml(String content) {
    	Hashtable h;	// the tag parameters
	HtmlRewriter hr = new HtmlRewriter(content);
	while (hr.nextTag()) {
	    String tag = hr.getTag().toLowerCase();
	    String param = (String) tagMap.get(tag);
	    if (param == null) {
		continue;
	    }
	    String value = hr.get(param);
	    if (value == null) {
		continue;
	    }
	    String fixed = convertString(value);
	    if (fixed != null) {
		hr.put(param, fixed);
	    }
	}
	return hr.toString();
    }

    /**
     * Rewrite a url inside a tag parameter.
     * 
     * @param fix	The value of the tag to be rewritten (fixed)
     * @return		null of the existing value is OK,
     *			otherwise the new value is returned
     */

    public String convertString(String fix) {
	if (fix.startsWith("/")) {
	    if (prefix == null) {
		return null;
	    }
	    count++;
	    String result =  prefix + fix.substring(1);
	    log(fix + " -> " + result);
	    return result;
	}
	if (fix.startsWith("http://") && urlMap != null) {
	    if (fix.indexOf("/",7) == -1) {
		log("  adding missing /");
		fix += "/";
	    }
	    Enumeration e = urlMap.keys();
	    while (e.hasMoreElements()) {
		String match = (String) e.nextElement();
		log("   " +    fix + " <?> " + match);
		if (fix.startsWith(match)) {
		    count++;
		    String result = urlMap.get(match) +
			    fix.substring(match.length());
		    log(fix + " -> " + result);
		    return result;
		}
	    }
	}

	/*
	 * check the regexp patterns
	 */

	for (int i = 0; i<patternMap.size(); i+= 3) {
	    Regexp exp = (Regexp) patternMap.elementAt(i+2);
	    String replace = exp.sub(fix, (String) patternMap.elementAt(i+1));
	    if (replace != null) { 
		count++;
		return replace;
	    }
	}
	log("No mapping for (" + fix + ")");
	return null;
    }

    /**
     * diagnostic output
     */

    public void
    log(String message) {
        if (log) {
	    System.out.println("MapPage: " + message);
        }
    }
}
